<?php
/**
 * Hotel Management System - Configuration File
 * Contains database connection and helper functions
 */

// Database Configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'hotel_db');

// Application Settings
define('SITE_NAME', 'GreenDoors Hotel');
define('SITE_URL', 'http://localhost/hotel/');

// Session Configuration
ini_set('session.gc_maxlifetime', 3600);
session_start();

// Create PDO Connection
try {
    $pdo = new PDO(
        'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME,
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::MYSQL_ATTR_CHARSET => 'utf8mb4'
        ]
    );
} catch (PDOException $e) {
    die('Database connection failed: ' . htmlspecialchars($e->getMessage()));
}

// ==================== Authentication Functions ====================

/**
 * Check if an admin is logged in
 */
function isAdminLoggedIn() {
    return isset($_SESSION['admin_id']) && isset($_SESSION['admin_user']);
}

/**
 * Get current logged-in admin user
 */
function getAdminUser() {
    return isset($_SESSION['admin_user']) ? $_SESSION['admin_user'] : null;
}

/**
 * Redirect to login if not authenticated
 */
function redirectToLogin() {
    header('Location: ../admin/login.php');
    exit;
}

/**
 * Logout the current admin
 */
function adminLogout() {
    session_destroy();
    header('Location: ../admin/login.php');
    exit;
}

// ==================== Input Validation Functions ====================

/**
 * Sanitize user input
 */
function sanitizeInput($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate email address
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate phone number (basic format)
 */
function validatePhone($phone) {
    return preg_match('/^[0-9\-\+\(\)\s]{7,}$/', $phone);
}

/**
 * Validate date format (YYYY-MM-DD)
 */
function validateDate($date) {
    $d = DateTime::createFromFormat('Y-m-d', $date);
    return $d && $d->format('Y-m-d') === $date;
}

// ==================== Booking Functions ====================

/**
 * Generate unique booking reference
 */
function generateBookingReference() {
    return 'BK' . strtoupper(substr(str_shuffle('ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'), 0, 8)) . time();
}

/**
 * Calculate booking price
 */
function calculateBookingPrice($check_in, $check_out, $category_price, $num_rooms = 1) {
    $start = new DateTime($check_in);
    $end = new DateTime($check_out);
    $interval = $start->diff($end);
    $nights = $interval->days;
    return $nights * $category_price * $num_rooms;
}

/**
 * Get available rooms for a date range
 */
function getAvailableRooms($check_in, $check_out, $category_id = null) {
    global $pdo;
    
    $sql = "
        SELECT r.* FROM rooms r
        WHERE r.status = 'available'
        AND r.id NOT IN (
            SELECT room_id FROM bookings 
            WHERE status IN ('confirmed', 'checked_in')
            AND (
                (check_in_date <= :check_out AND check_out_date >= :check_in)
            )
        )
    ";
    
    $params = [
        ':check_in' => $check_in,
        ':check_out' => $check_out
    ];
    
    if ($category_id) {
        $sql .= " AND r.category_id = :category_id";
        $params[':category_id'] = $category_id;
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
}

// ==================== Database Helper Functions ====================

/**
 * Execute a SELECT query
 */
function dbSelect($query, $params = []) {
    global $pdo;
    try {
        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log('Database error: ' . $e->getMessage());
        return [];
    }
}

/**
 * Execute an INSERT, UPDATE, or DELETE query
 */
function dbExecute($query, $params = []) {
    global $pdo;
    try {
        $stmt = $pdo->prepare($query);
        return $stmt->execute($params);
    } catch (PDOException $e) {
        error_log('Database error: ' . $e->getMessage());
        return false;
    }
}

/**
 * Get last insert ID
 */
function dbLastInsertId() {
    global $pdo;
    return $pdo->lastInsertId();
}

// ==================== Logging Functions ====================

/**
 * Log admin activity
 */
function logActivity($action, $description, $entity_type = null, $entity_id = null) {
    global $pdo;
    
    $admin_id = isAdminLoggedIn() ? $_SESSION['admin_id'] : null;
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? '';
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs (admin_id, action, description, entity_type, entity_id, ip_address)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$admin_id, $action, $description, $entity_type, $entity_id, $ip_address]);
    } catch (PDOException $e) {
        error_log('Activity log error: ' . $e->getMessage());
    }
}

// ==================== Response Functions ====================

/**
 * Send JSON response
 */
function respondJson($success, $message = '', $data = null) {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data
    ]);
    exit;
}

/**
 * Redirect with message
 */
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['message'] = $message;
    $_SESSION['message_type'] = $type;
    header('Location: ' . $url);
    exit;
}

/**
 * Get and clear flash message
 */
function getFlashMessage() {
    if (isset($_SESSION['message'])) {
        $message = $_SESSION['message'];
        $type = $_SESSION['message_type'] ?? 'info';
        unset($_SESSION['message']);
        unset($_SESSION['message_type']);
        return ['message' => $message, 'type' => $type];
    }
    return null;
}

// ==================== Utility Functions ====================

/**
 * Format currency
 */
function formatCurrency($amount) {
    return '$' . number_format($amount, 2);
}

/**
 * Format date
 */
function formatDate($date, $format = 'M d, Y') {
    return date($format, strtotime($date));
}

/**
 * Get status badge
 */
function getStatusBadge($status) {
    $badges = [
        'available' => '<span class="badge bg-success">Available</span>',
        'occupied' => '<span class="badge bg-danger">Occupied</span>',
        'maintenance' => '<span class="badge bg-warning">Maintenance</span>',
        'blocked' => '<span class="badge bg-secondary">Blocked</span>',
        'pending' => '<span class="badge bg-info">Pending</span>',
        'confirmed' => '<span class="badge bg-success">Confirmed</span>',
        'checked_in' => '<span class="badge bg-primary">Checked In</span>',
        'checked_out' => '<span class="badge bg-secondary">Checked Out</span>',
        'cancelled' => '<span class="badge bg-danger">Cancelled</span>'
    ];
    
    return $badges[$status] ?? '<span class="badge bg-secondary">' . ucfirst($status) . '</span>';
}

/**
 * Generate ratings display
 */
function generateStars($rating) {
    $stars = '';
    for ($i = 1; $i <= 5; $i++) {
        $stars .= $i <= $rating ? '<i class="fas fa-star"></i>' : '<i class="far fa-star"></i>';
    }
    return $stars;
}
