<?php
require_once __DIR__ . '/../includes/config.php';

if (!isAdminLoggedIn()) {
    redirectToLogin();
}

$admin = getAdminUser();

// Fetch dashboard statistics
try {
    // Total bookings
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM bookings");
    $total_bookings = $stmt->fetch()['total'];
    
    // Today's check-ins
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM bookings WHERE DATE(check_in_date) = CURDATE()");
    $today_checkins = $stmt->fetch()['total'];
    
    // Available rooms
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM rooms WHERE status = 'available'");
    $available_rooms = $stmt->fetch()['total'];
    
    // Total revenue
    $stmt = $pdo->query("SELECT SUM(amount) as total FROM payments WHERE payment_status = 'completed'");
    $total_revenue = $stmt->fetch()['total'] ?? 0;
    
    // Total guests
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM guests");
    $total_guests = $stmt->fetch()['total'];
    
    // Pending bookings
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM bookings WHERE status = 'pending'");
    $pending_bookings = $stmt->fetch()['total'];
    
    // Occupied rooms
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM rooms WHERE status = 'occupied'");
    $occupied_rooms = $stmt->fetch()['total'];
    
    // Recent bookings
    $stmt = $pdo->query("
        SELECT b.*, g.first_name, g.last_name, rc.category_name
        FROM bookings b
        JOIN guests g ON b.guest_id = g.id
        JOIN rooms r ON b.room_id = r.id
        JOIN room_categories rc ON r.category_id = rc.id
        ORDER BY b.created_at DESC
        LIMIT 5
    ");
    $recent_bookings = $stmt->fetchAll();
    
} catch (Exception $e) {
    $error = 'Failed to load dashboard data';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - GreenDoors Hotel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin-style.css">
</head>
<body>
    <div class="wrapper">
        <!-- Sidebar Navigation -->
        <nav class="sidebar">
            <div class="sidebar-brand">
                <i class="fas fa-cube"></i>
                <span>GreenDoors</span>
            </div>
            
            <ul class="sidebar-nav">
                <li class="nav-item active">
                    <a class="nav-link" href="index.php">
                        <i class="fas fa-chart-line"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="rooms.php">
                        <i class="fas fa-door-open"></i>
                        <span>Rooms</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="bookings.php">
                        <i class="fas fa-calendar-check"></i>
                        <span>Bookings</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="guests.php">
                        <i class="fas fa-users"></i>
                        <span>Guests</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="payments.php">
                        <i class="fas fa-money-bill"></i>
                        <span>Payments</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="reviews.php">
                        <i class="fas fa-star"></i>
                        <span>Reviews</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="categories.php">
                        <i class="fas fa-th"></i>
                        <span>Categories</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="admin-users.php">
                        <i class="fas fa-user-tie"></i>
                        <span>Admin Users</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" href="activity-logs.php">
                        <i class="fas fa-history"></i>
                        <span>Activity Logs</span>
                    </a>
                </li>
            </ul>
            
            <div class="sidebar-footer">
                <a class="nav-link" href="logout.php">
                    <i class="fas fa-sign-out-alt"></i>
                    <span>Logout</span>
                </a>
            </div>
        </nav>

        <!-- Main Content -->
        <div class="main-content">
            <!-- Top Navigation Bar -->
            <nav class="navbar navbar-expand-lg navbar-light bg-white border-bottom">
                <div class="container-fluid">
                    <span class="navbar-text">Welcome, <strong><?php echo htmlspecialchars($admin['full_name'] ?? $admin['username']); ?></strong></span>
                    <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                        <span class="navbar-toggler-icon"></span>
                    </button>
                    <div class="collapse navbar-collapse" id="navbarNav" style="justify-content: flex-end;">
                        <div class="d-flex align-items-center gap-3">
                            <div class="admin-profile">
                                <img src="https://i.pravatar.cc/100?u=<?php echo $admin['id']; ?>" alt="Profile">
                                <div>
                                    <p class="mb-0"><?php echo htmlspecialchars($admin['full_name'] ?? $admin['username']); ?></p>
                                    <small class="text-muted"><?php echo ucfirst($admin['role']); ?></small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </nav>

            <!-- Page Content -->
            <div class="page-content">
                <div class="page-header">
                    <h1>Dashboard</h1>
                    <p class="text-muted">Overview of your hotel management system</p>
                </div>

                <!-- Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                                <i class="fas fa-calendar-check"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Total Bookings</p>
                                <h3><?php echo $total_bookings; ?></h3>
                                <small class="text-muted">All time bookings</small>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                                <i class="fas fa-clock"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Today's Check-ins</p>
                                <h3><?php echo $today_checkins; ?></h3>
                                <small class="text-muted">Expected today</small>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
                                <i class="fas fa-door-open"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Available Rooms</p>
                                <h3><?php echo $available_rooms; ?></h3>
                                <small class="text-muted">Ready for booking</small>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);">
                                <i class="fas fa-money-bill"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Total Revenue</p>
                                <h3><?php echo formatCurrency($total_revenue); ?></h3>
                                <small class="text-muted">Completed payments</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Additional Statistics -->
                <div class="row mb-4">
                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);">
                                <i class="fas fa-users"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Total Guests</p>
                                <h3><?php echo $total_guests; ?></h3>
                                <small class="text-muted">Registered guests</small>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #a8edea 0%, #fed6e3 100%);">
                                <i class="fas fa-hourglass"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Pending Bookings</p>
                                <h3><?php echo $pending_bookings; ?></h3>
                                <small class="text-muted">Awaiting confirmation</small>
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-3 col-md-6 mb-4">
                        <div class="stat-card">
                            <div class="stat-icon" style="background: linear-gradient(135deg, #ff9a56 0%, #ff6a88 100%);">
                                <i class="fas fa-hotel"></i>
                            </div>
                            <div class="stat-content">
                                <p class="stat-label">Occupied Rooms</p>
                                <h3><?php echo $occupied_rooms; ?></h3>
                                <small class="text-muted">Currently occupied</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent Bookings -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Recent Bookings</h5>
                    </div>
                    <div class="card-body table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Booking ID</th>
                                    <th>Guest Name</th>
                                    <th>Room Type</th>
                                    <th>Check-in</th>
                                    <th>Check-out</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_bookings as $booking): ?>
                                    <tr>
                                        <td><code><?php echo htmlspecialchars($booking['booking_reference']); ?></code></td>
                                        <td><?php echo htmlspecialchars($booking['first_name'] . ' ' . $booking['last_name']); ?></td>
                                        <td><?php echo htmlspecialchars($booking['category_name']); ?></td>
                                        <td><?php echo formatDate($booking['check_in_date'], 'M d, Y'); ?></td>
                                        <td><?php echo formatDate($booking['check_out_date'], 'M d, Y'); ?></td>
                                        <td><?php echo getStatusBadge($booking['status']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="../assets/js/admin-script.js"></script>
</body>
</html>
