# GreenDoors Hotel Management System

A complete hotel management system built with PHP, MySQL, and Bootstrap, featuring a professional admin panel and a modern public website with a premium green/olive color scheme.

## Features

### 🏨 Public Website
- **Modern Hero Section**: Attractive landing page with circular room images
- **Room Booking System**: Easy-to-use booking form with date selection
- **Room Categories Showcase**: Display all available room types with pricing
- **Testimonials Section**: Customer reviews and ratings
- **Newsletter Subscription**: Email subscription feature
- **Responsive Design**: Mobile-friendly interface with green/olive theme
- **Featured Hotels**: Showcase premium rooms with ratings and pricing

### 👨‍💼 Admin Panel
- **Secure Login System**: Bcrypt password hashing, role-based access control
- **Dashboard**: Real-time statistics and key metrics
- **Room Management**: CRUD operations for room categories and rooms
- **Booking Management**: View, update, and manage bookings
- **Guest Management**: Customer information and booking history
- **Payment Tracking**: Revenue reports and payment status
- **Review Management**: Customer ratings and feedback
- **Activity Logging**: Complete audit trail of admin actions
- **Admin User Management**: Create and manage admin accounts

### 🗄️ Database
- **10 Optimized Tables**: Normalized MySQL schema with relationships
- **Sample Data**: Pre-loaded room categories, amenities, and admin user
- **Prepared Statements**: SQL injection protection
- **Timestamps**: Automatic created_at and updated_at fields

## Project Structure

```
hotel/
├── admin/                      # Admin panel files
│   ├── login.php              # Admin login page
│   ├── index.php              # Dashboard
│   ├── rooms.php              # Room management
│   ├── bookings.php           # Booking management
│   ├── guests.php             # Guest management
│   ├── payments.php           # Payment tracking
│   ├── reviews.php            # Review management
│   ├── categories.php         # Room categories
│   ├── admin-users.php        # Admin user management
│   ├── activity-logs.php      # Activity logs viewer
│   └── logout.php             # Logout handler
│
├── public/                     # Public website files
│   ├── index.php              # Home page
│   ├── rooms.php              # Room listing
│   ├── booking.php            # Booking form
│   └── contact.php            # Contact page
│
├── includes/                   # Core files
│   └── config.php             # Database config & helper functions
│
├── database/                   # Database files
│   └── hotel_db.sql           # MySQL schema and sample data
│
├── assets/
│   ├── css/
│   │   ├── admin-style.css    # Admin panel styling
│   │   └── public-style.css   # Public website styling
│   ├── js/
│   │   └── admin-script.js    # Admin panel JavaScript
│   └── images/                # Image storage
│
├── api/                        # Future API endpoints
│   ├── rooms.php              # Room API
│   └── bookings.php           # Booking API
│
└── README.md                   # This file
```

## Installation & Setup

### Requirements
- PHP 7.4+
- MySQL 5.7+
- Apache/Nginx web server
- Modern web browser

### Step 1: Install Files
1. Extract the project files to your web server directory (e.g., `htdocs/` for XAMPP)
2. Ensure the folder structure is preserved

### Step 2: Create Database
1. Open phpMyAdmin or MySQL command line
2. Create a new database named `hotel_db`
3. Import the database schema:
   ```bash
   mysql -u root -p hotel_db < database/hotel_db.sql
   ```

### Step 3: Configure Database Connection
1. Open `includes/config.php`
2. Update database credentials if needed:
   ```php
   define('DB_HOST', 'localhost');
   define('DB_USER', 'root');
   define('DB_PASS', '');
   define('DB_NAME', 'hotel_db');
   ```

### Step 4: Access the Application
- **Public Website**: `http://localhost/hotel/public/index.php`
- **Admin Panel**: `http://localhost/hotel/admin/login.php`

## Demo Credentials

| Field | Value |
|-------|-------|
| Username | admin |
| Password | admin123 |

## Color Scheme

The application uses a premium green/olive color palette:

| Color | Hex Value | Usage |
|-------|-----------|-------|
| Primary | `#4a7c59` | Navigation, buttons, accents |
| Secondary | `#6b9a7a` | Hover states, gradients |
| Dark | `#2d5a3d` | Text, dark elements |
| Accent | `#a4c96f` | Highlights, icons |
| Light Background | `#f8faf6` | Page backgrounds |

## API Endpoints (Development Ready)

### Rooms API
```
GET /api/rooms.php - Get all rooms
GET /api/rooms.php?category_id=1 - Get rooms by category
POST /api/rooms.php - Create new room
PUT /api/rooms.php - Update room
DELETE /api/rooms.php - Delete room
```

### Bookings API
```
GET /api/bookings.php - Get all bookings
POST /api/bookings.php - Create new booking
PUT /api/bookings.php - Update booking
DELETE /api/bookings.php - Cancel booking
```

## Key Features Explained

### Authentication System
- Secure login with session management
- Password hashing using bcrypt
- Role-based access control (Admin, Manager, Staff)
- Automatic logout with session timeout

### Booking Management
- Check date and room availability validation
- Automatic price calculation based on room category and stay duration
- Booking status tracking (Pending, Confirmed, Checked-in, Checked-out, Cancelled)
- Special requests and guest preferences

### Payment Processing
- Multiple payment methods support
- Payment status tracking (Pending, Completed, Failed, Refunded)
- Revenue reporting and analytics
- Transaction ID recording

### Activity Logging
- Complete audit trail of all admin actions
- User IP tracking
- Timestamp recording
- Action categorization

### Room Management
- Room categories with base pricing
- Occupancy limits per room type
- Room status tracking (Available, Occupied, Maintenance, Blocked)
- Amenities association

## Helper Functions (in config.php)

### Authentication Functions
- `isAdminLoggedIn()` - Check if admin is logged in
- `getAdminUser()` - Get current logged-in user
- `redirectToLogin()` - Redirect to login if not authenticated
- `adminLogout()` - Clear session and logout

### Input Validation Functions
- `sanitizeInput($input)` - Sanitize user input
- `validateEmail($email)` - Validate email format
- `validatePhone($phone)` - Validate phone number
- `validateDate($date)` - Validate date format

### Booking Functions
- `generateBookingReference()` - Generate unique booking ID
- `calculateBookingPrice()` - Calculate total booking cost
- `getAvailableRooms()` - Get available rooms for date range

### Database Functions
- `dbSelect($query, $params)` - Execute SELECT query
- `dbExecute($query, $params)` - Execute INSERT/UPDATE/DELETE
- `dbLastInsertId()` - Get last inserted ID

### Utility Functions
- `formatCurrency($amount)` - Format currency display
- `formatDate($date, $format)` - Format date
- `getStatusBadge($status)` - Get HTML status badge
- `generateStars($rating)` - Generate star rating HTML

## Security Features

✅ **SQL Injection Protection**: All queries use prepared statements
✅ **Password Security**: Bcrypt hashing with salt
✅ **Session Management**: Secure session handling
✅ **Input Sanitization**: HTML escaping and trimming
✅ **CSRF Protection**: Form validation
✅ **Error Handling**: Graceful error messages
✅ **Activity Logging**: Complete audit trail

## Responsive Design

The application is fully responsive and works on:
- Desktop browsers (1200px+)
- Tablets (768px - 1199px)
- Mobile devices (320px - 767px)

## File Permissions

Ensure proper file permissions:
```bash
chmod 755 hotel/          # Directory
chmod 644 hotel/*.php     # PHP files
chmod 644 hotel/assets/*  # Asset files
```

## Browser Support

- Chrome (latest)
- Firefox (latest)
- Safari (latest)
- Edge (latest)
- Mobile browsers

## Customization Guide

### Changing Color Scheme
Edit CSS variables in `assets/css/public-style.css`:
```css
:root {
    --primary-color: #4a7c59;
    --secondary-color: #6b9a7a;
    --accent-color: #a4c96f;
    /* ... other colors ... */
}
```

### Adding New Admin Modules
1. Create new PHP file in `/admin/`
2. Start with authentication check:
   ```php
   <?php
   require_once __DIR__ . '/../includes/config.php';
   if (!isAdminLoggedIn()) {
       redirectToLogin();
   }
   ```
3. Add navigation link in sidebar
4. Implement CRUD operations as needed

### Customizing Room Categories
Edit room categories in the admin panel or database:
```sql
INSERT INTO room_categories (category_name, base_price, max_occupancy, amenities)
VALUES ('Suite', 300.00, 4, 'WiFi, TV, Kitchenette');
```

## Database Schema Overview

### admin_users
Stores admin account information with role-based access

### rooms
Physical rooms with status, floor, category assignment

### room_categories
Room types (Single, Double, Suite) with pricing

### guests
Guest information and contact details

### bookings
Room reservations with dates and status

### payments
Payment records and transaction information

### reviews
Guest ratings and feedback

### amenities
Available room features and services

### room_amenities
Link between rooms and amenities

### activity_logs
Complete audit trail of admin actions

## Performance Tips

1. **Database Optimization**
   - Indexes on frequently searched fields
   - Use EXPLAIN to analyze queries
   - Regular table maintenance

2. **Caching**
   - Consider implementing query caching
   - Cache static content
   - Use browser caching headers

3. **Code Optimization**
   - Minimize database queries
   - Use pagination for large datasets
   - Lazy load images

## Troubleshooting

### Database Connection Error
- Check database credentials in `config.php`
- Ensure MySQL server is running
- Verify database `hotel_db` exists

### Login Issues
- Check admin user in database
- Reset password using database:
  ```sql
  UPDATE admin_users SET password = '$2y$10$...' WHERE id = 1;
  ```

### Room Availability Issues
- Check booking overlap logic in `getAvailableRooms()`
- Verify room status in database
- Review booking dates

### CSS/Styling Issues
- Clear browser cache (Ctrl+F5)
- Check Bootstrap CDN connection
- Verify FontAwesome CDN availability

## Future Enhancements

- [ ] Email notifications for bookings
- [ ] SMS integration
- [ ] Payment gateway integration (Stripe, PayPal)
- [ ] Advanced reporting and analytics
- [ ] Housekeeping management module
- [ ] Maintenance scheduling system
- [ ] Multi-language support
- [ ] API documentation (Swagger/OpenAPI)
- [ ] Mobile app
- [ ] Two-factor authentication

## Support & Maintenance

For issues or questions:
1. Check the troubleshooting section
2. Review error logs in browser console
3. Check PHP error logs

## License

This project is open source and available under the MIT License.

## Version History

**v1.0.0** (2024)
- Initial release
- Complete admin panel
- Public website with modern design
- Database schema
- Authentication system
- Booking management
- Payment tracking
- Activity logging

## Contact

For support and inquiries, please contact:
- Email: info@greendoors.com
- Phone: +1 (555) 123-4567
- Website: https://greendoors.example.com

---

**Made with ❤️ for hotel management excellence**
